/*
 * @brief Secondary loader SPI host interface handler
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "sl_common.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

static LPC_SPI_T *savedPort;
static LPC_SPI_T *pSPIArray[1] = {
	LPC_SPI0
};



extern uint8_t Chip_SPIS_FindSSEL(LPC_SPI_T *pSPI, uint32_t data);

/* Possible function selections for SPI mapped pins in IOCON */
#define SPIPURP_SCK         0
#define SPIPURP_MOSI        1
#define SPIPURP_MISO        2
#define SPIPURP_SSEL        3
typedef struct {
	uint8_t     spiNum;			/* SPin number, 0 or 1 */
	uint8_t     port;			/* Port */
	uint8_t     pin;			/* Pin */
	uint8_t     func;			/* IOCON function number */
	uint8_t     purpose;		/* 0 = SCK, 1 = MOSI, 2 = MISO, 3 = SSEL */
} SL_SPIFNUCMAP_T;
#define SPIPURP_MAPMAPPINS  17
static const SL_SPIFNUCMAP_T spiMappedFuncs[SPIPURP_MAPMAPPINS] = {


};

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

volatile SPIS_XFER_T spiSlaveXfer;
extern int buffXferPending;
extern uint32_t g_spiTxPending;
int spi_tx_index=0;

/* DMA descriptors must be aligned to 16 bytes */
#if defined(__CC_ARM)	/* Keil support */
//__align(16) static DMA_CHDESC_T dmaSPISTxDesc[1];
//__align(16) static DMA_CHDESC_T dmaSPISRxDesc[1];
#elif defined(__ICCARM__)	/* IAR support */
#pragma data_alignment=16
static DMA_CHDESC_T dmaSPISTxDesc[1];
#pragma data_alignment=16
static DMA_CHDESC_T dmaSPISRxDesc[1];
#elif defined( __GNUC__ )	/* GCC support */
static DMA_CHDESC_T dmaSPISTxDesc[1] __attribute__ ((aligned(16)));
static DMA_CHDESC_T dmaSPISRxDesc[1] __attribute__ ((aligned(16)));
#endif /* defined (__GNUC__) */

/*****************************************************************************
 * Private functions
 ****************************************************************************/



#define SPI_READDMA_XFERSIZE    (SL_HOSTIFBUFFSIZE + 1)

/* SPI slave select assertion callback function */
static void SPISlaveAssert(SPIS_XFER_T *pSlaveXfer)
{
	Hostif_DeAssertIRQ();
}

/* SPI slave send data callback function */
static void SPISlaveSendData(SPIS_XFER_T *pSlaveXfer)
{
	/* Do nothing */
}

/* SPI slave receive data callback function */
static void SPISlaveRecvData(SPIS_XFER_T *pSlaveXfer)
{
	/* Do nothing */
}

/* SPI slave select de-assertion callback function */
static void SPISlaveDeAssert(SPIS_XFER_T *pSlaveXfer)
{

//	/* check if we sending response */
	//如果数据已经接收完成，且相应的命令已经执行完成，给主机回应，主机是不是接收到相应之后才会拉高CS的？
	if (g_spiTxPending) {
		//g_spiTxPending = 0;
		startHostIfSPI();
		spi_tx_index++;
	}
	else {

	}
	//重新配置接收和发送数组
 // startHostIfSPI();
	//表示一个数据包已经接收完成
	buffXferPending = 1;
}

/* SPI slave driver callbacks */
static const SPIS_CALLBACKS_T spiSlaveCallbacks = {
	&SPISlaveAssert,
	&SPISlaveSendData,
	&SPISlaveRecvData,
	&SPISlaveDeAssert
};

/* Maps a pin's IOCON function to it's SPI number, port, pin, and purpose */
static void spiMapPin(int spiNum, SL_PORTPIN_T *pSpiPortPin, uint8_t purp)
{
	int i = 0;
	uint32_t func;
	bool mapped = false;

	while ((mapped == false) && (i < SPIPURP_MAPMAPPINS)) {
		if (spiMappedFuncs[i].spiNum == spiNum) {
			if (spiMappedFuncs[i].purpose == purp) {
				if (pSpiPortPin->port == spiMappedFuncs[i].port) {
					if (pSpiPortPin->pin == spiMappedFuncs[i].pin) {
						/* Matched function to pin */
						func = (uint32_t) spiMappedFuncs[i].func;

						break;
					}
				}
			}
		}

		i++;
	}
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

void setupMuxingSPI(int spiNum)
{
	/* Pin selection based on port. Note MISO is not setup here, as it is
	   setup later after A5 packet processing (also ok for app provided
	   MISO). */
	spiMapPin(spiNum, &slIfConfig.hostMosiPortPin, SPIPURP_MOSI);
	spiMapPin(spiNum, &slIfConfig.hostSselPortPin, SPIPURP_SSEL);
	spiMapPin(spiNum, &slIfConfig.hostSckPortPin, SPIPURP_SCK);
}

/* Sets up pinmuxing for an SPI MISO pin only */
void setupMuxingSPIMiso(int spiNum)
{
	spiMapPin(spiNum, &slIfConfig.hostMisoPortPin, SPIPURP_MISO);
}

void setupInterfaceSPI(int spiNum)
{
	savedPort = pSPIArray[0];//--Chooce as LPC_SPI0

	/* Setup slave controller SSEL for active low select */
	Chip_SPI_SetCSPolLow(savedPort, 0);

	/* Setup slave controller for 8-bit transfers */
	Chip_SPI_SetControlInfo(savedPort, 8, 0);

	/* Enable SPI0 */
	Chip_SPI_Enable(savedPort);

	Chip_SPI_EnableInts(savedPort, (SPI_INTENSET_RXDYEN | SPI_INTENSET_RXOVEN |
									SPI_INTENSET_TXUREN | SPI_INTENSET_SSAEN | SPI_INTENSET_SSDEN));

	/* Setup slave transfer callbacks in the transfer descriptor */
	spiSlaveXfer.pCB = &spiSlaveCallbacks;																					//-- -- -- 
}

void startHostIfSPI(void)
{	
	/*pTXData8 point to sendBuff.buffer[1], because sendBuff.buffer[0] will be preload to TXData now*/
	spiSlaveXfer.pTXData8 = &sendBuff.buffer[1];
	spiSlaveXfer.txCount = SL_HOSTIFBUFFSIZE;
	spiSlaveXfer.pRXData8 = &recvBuff.buffer[0];
	spiSlaveXfer.rxCount = SL_HOSTIFBUFFSIZE;
	spiSlaveXfer.dataRXferred = 0;
	spiSlaveXfer.dataTXferred = 0;

	/* Flush any pre loaded SPI data */
	Chip_SPI_FlushFifos(LPC_SPI0);
	
	/*Preload data to TXDATA to response master*/
	Chip_SPI_WriteTXData(LPC_SPI0, sendBuff.buffer[0]);

}

/* SPI slave transfer state change handler */
void SPI_Slave_Rx_Blocking(LPC_SPI_T *pSPI, SPIS_XFER_T *xfer)
{

	//在CS下降沿到来时，调用此函数；在CS上升沿到来之前，一直停留在while 循环中接收数据
	while((Chip_SPI_GetStatus(pSPI) & SPI_STAT_SSD) == 0)
	{
		while ((Chip_SPI_GetStatus(pSPI) & SPI_STAT_RXRDY) != 0) {
			/* Get raw data and status */
			*xfer->pRXData8  = Chip_SPI_ReadRawRXFifo(pSPI);
			xfer->pRXData8++;					
			xfer->dataRXferred++;		
		}
	}
}

void SPI_Slave_Tx_Blocking(LPC_SPI_T *pSPI, SPIS_XFER_T *xfer)
{
	uint32_t  data;

	//等待CS上升沿到来
	while((Chip_SPI_GetStatus(pSPI) & SPI_STAT_SSD) == 0)
	{
		while ((Chip_SPI_GetStatus(pSPI) & SPI_STAT_TXRDY) != 0) {
			data = (uint32_t) *xfer->pTXData8;
			Chip_SPI_WriteTXData(pSPI, data);
			xfer->pTXData8++;
			xfer->dataTXferred++;								
		}		
	}					
}

/* SPI slave transfer state change handler */
uint32_t SPI_Slave_Handler(LPC_SPI_T *pSPI, SPIS_XFER_T *xfer)
{
	uint32_t staterr;
	//uint8_t flen;

	//flen = ((pSPI->TXCTRL & ~SPI_TXCTRL_RESERVED) >> 24) & 0xF;

	staterr = Chip_SPI_GetStatus(pSPI) & (SPI_STAT_RXOV | SPI_STAT_TXUR);
	if (staterr != 0) {
		Chip_SPI_ClearStatus(pSPI, staterr);
	}

	/* Slave assertion, CS pin is pulled low */						//--??--CS Low is working state
	if ((Chip_SPI_GetStatus(pSPI) & SPI_STAT_SSA) != 0) { //--??--Slave Select Assert. 
		Chip_SPI_ClearStatus(pSPI, SPI_STAT_SSA);

		/* Determine SPI select. Read the data FIFO to get the slave number. Data
		   should not be in the receive FIFO yet, only the statuses */
		xfer->sselNum = Chip_SPIS_FindSSEL(pSPI, Chip_SPI_ReadRawRXFifo(pSPI));

		/* SSEL assertion callback, if CS is pulled low, pull IRQ high */		//--Not ready to response to Master now.
		xfer->pCB->slaveXferCSAssert(xfer);
		
		/*If CS pin is low and g_spiTxPending==0, receive data */           //--receive!
		if(g_spiTxPending==0)
		{
			xfer->dataRXferred=0;	
			SPI_Slave_Rx_Blocking(LPC_SPI0,xfer);
		}
		
		/*Response data to SPI master*/
		if(g_spiTxPending==1)
		{
			xfer->dataTXferred=0;
			SPI_Slave_Tx_Blocking(LPC_SPI0,xfer);
			g_spiTxPending=0;
		}
	}

	/* Slave de-assertion, if the rising edge of CS is detected---receive data completed or response data completed*/
	if ((Chip_SPI_GetStatus(pSPI) & SPI_STAT_SSD) != 0) {
		Chip_SPI_ClearStatus(pSPI, SPI_STAT_SSD);
		//deasserted = true;
		
		/*startHostSPI, configure spiSlaveXfer structure for new command or new response*/
		xfer->pCB->slaveXferCSDeAssert(xfer);
	}



	return staterr;

}



/* SPI host interface processing loop */
void loopHostIfSPI(int spiNum)
{
	SPI_Slave_Handler(LPC_SPI0, (SPIS_XFER_T *)&spiSlaveXfer);      //--??-- SPI slave transfer state change handler
	recvBuff.bytes=spiSlaveXfer.dataRXferred;
}

/* Shut down a SPI slave interface */
void shutdownInterfaceSPI(int spiNum)
{
	
	Chip_SPI_Disable(LPC_SPI0);
	Chip_SPI_DeInit(LPC_SPI0);
}
